function generate_array(size, fun) {
    return Array(size).fill(0).map((_, i) => fun(i));
}

function generate_matrix_2d(sx, sy, fun) {
    return generate_array(sx, x => generate_array(sy, y => fun(x, y)));
}

function random_noise_2d(w, h, n) {
    return generate_matrix_2d(w, h, _ => generate_array(n, _ => Math.random()));
}
VECNOISE16_2D_0 = random_noise_2d(16, 16, 2);
VECNOISE16_2D_1 = random_noise_2d(16, 16, 2);
VECNOISE16_2D_2 = random_noise_2d(16, 16, 2);


function perlin_noise_2d(x, y, vecnoise) {
    let [w, h] = [vecnoise.length, vecnoise[0].length];
    let [whole_x, whole_y] = [Math.floor(x), Math.floor(y)];
    let [part_x, part_y] = [x - whole_x, y - whole_y];
    let [d1, d2, d3, d4] = [
        vecSubtract([x, y], [whole_x, whole_y]),
        vecSubtract([x, y], [whole_x+1, whole_y]),
        vecSubtract([x, y], [whole_x+1, whole_y+1]),
        vecSubtract([x, y], [whole_x, whole_y+1]),
    ]
    let [q1, q2, q3, q4] = [
        dotProduct(d1, vecnoise[whole_x][whole_y]),
        dotProduct(d2, vecnoise[whole_x+1][whole_y]),
        dotProduct(d3, vecnoise[whole_x+1][whole_y+1]),
        dotProduct(d4, vecnoise[whole_x][whole_y+1]),
    ]

    let r1 = interpolate(q1, q2, part_x);
    let r2 = interpolate(q4, q3, part_x);
    let r = interpolate(r1, r2, part_y);

    return r;
}


function random_coord(scales, dim=3) {
    coord = Array(dim);
    for (let i = 0; i < dim; ++i) {
        coord[i] = Math.random() * scales[i];
    }
    return coord;
}


function spaceflight(canvas, ctx, MID, {timescale = 1.0} = {}) {
    let stars = Array(1000);
    let w = canvas.width; let h = canvas.height;
    let space_scale = [w * 20, h * 20, 41];
    for (let i = 0; i < stars.length; ++i) {
        stars[i] = random_coord(space_scale);
    }
    let controls = {
        beat: 0.0
    }
    loop = (t) => {
        stars.forEach((s) => {
            let z = (-s[2] - t*timescale) % space_scale[2] + space_scale[2];
            if (z > 0) {
                let scale = 1. / z;
                let x = (s[0] - space_scale[0]/2) * scale;
                let y = (s[1] - space_scale[1]/2) * scale;
                draw_circle(ctx, x + MID[0], y + MID[1], scale + controls.beat/z, '#FFFFFF', '#FFFFFF');
            }
        });
    }
    return [loop, controls];
}


function plasma_color(x, y, w, h, t, tex1, tex2, tex3) {
    let perlins = [
        tex1[Math.ceil(x / w * (tex1.length-1))][Math.ceil(y / h * (tex1[0].length-1))],
        tex2[Math.ceil(x / w * (tex2.length-1))][Math.ceil(y / h * (tex2[0].length-1))],
        tex3[Math.ceil(x / w * (tex3.length-1))][Math.ceil(y / h * (tex3[0].length-1))],
    ];
    let tt = (t) % 3;
    let ttw = Math.floor(tt);
    let ttf = tt - ttw;
    let i1 = ttw;
    let i2 = (ttw + 1) % 3;
    let perlin = interpolate(perlins[i1], perlins[i2], smoothstep(ttf));
    let c = Math.sin(perlin*40.*(1.0+.2*Math.cos(t*.001)))*255.;
    return [c, c, c];
}


function plasma(canvas, ctx, MID, {
        timescale = 1.0
    } = {})
{
    let width = 32;
    let height = 32;
    let sw = canvas.width/width;
    let sh = canvas.height/height;

    let [tw, th] = [width*2, height*2]
    let perlin_text_1 = generate_matrix_2d(tw, th, (x, y) => perlin_noise_2d(x/tw*3, y/th*3, VECNOISE16_2D_0));
    let perlin_text_2 = generate_matrix_2d(tw, th, (x, y) => perlin_noise_2d(x/tw*3, y/th*3, VECNOISE16_2D_1));
    let perlin_text_3 = generate_matrix_2d(tw, th, (x, y) => perlin_noise_2d(x/tw*3, y/th*3, VECNOISE16_2D_2));

    return [(t) => {
        for (let x = 0; x < width; ++x) {
            for (let y = 0; y < height; ++y) {
                // "pixel" rectangle points
                [x1, y1] = [x * sw, y * sh];
                [x2, y2] = [(x+1) * sw, (y) * sh];
                [x3, y3] = [(x+1) * sw, (y+1) * sh];
                [x4, y4] = [(x) * sw, (y+1) * sh];
                // "pixel" triangle sample points
                [sx1, sy1] = [x2, y2];
                [sx2, sy2] = [x4, y4];
                // triangle colors at sample
                c1 = plasma_color(sx1, sy1, canvas.width, canvas.height, t*timescale, perlin_text_1, perlin_text_2, perlin_text_3);
                c1str = 'rgb(' + c1[0] + ',' + c1[1] + ',' + c1[2] + ')';
                c2 = plasma_color(sx2, sy2, canvas.width, canvas.height, t*timescale, perlin_text_1, perlin_text_2, perlin_text_3);
                c2str = 'rgb(' + c2[0] + ',' + c2[1] + ',' + c2[2] + ')';
                // draw triangles
                draw_poly(ctx, [[x1, y1], [x2, y2], [x3, y3]], c1str, c1str, 1);
                draw_poly(ctx, [[x1, y1], [x4, y4], [x3, y3]], c2str, c2str, 1);
            }
        }
    }, {}];
}


function web(canvas, ctx, MID, {
    timescale = 1.,
    lt = 0,
    r = 70,
    rmin = .2,
    n_points = 64,
    vel_mult = 100.
    } = {})
{
    let [width, height] = [canvas.width, canvas.height];
    let points = generate_array(n_points, _ => [width*Math.random(), height*Math.random()]);
    let velocities = generate_array(n_points, _ => generate_array(2, _ => vel_mult * Math.random() - vel_mult/2));
    let controls = {
        beat: 0.
    };

    return [(t) => {
        r = 100 + 64. * controls.beat
        let dt = (t - lt)*timescale;
        lt = t;
        points.forEach((p1, i) => {
            let vel = velocities[i];
            p1[0] += vel[0]*dt;
            p1[1] += vel[1]*dt;
            if (p1[0] > width + rmin) p1[0] = -rmin;
            if (p1[0] < -rmin) p1[0] = width + rmin;
            if (p1[1] > height + rmin) p1[1] = -rmin;
            if (p1[1] < -rmin) p1[1] = height + rmin;
        });
        points.forEach((p1, i) => {
            let n_pairs = 0;
            points.forEach(p2 => {
                let d = vecDist(p1, p2);
                if (d < r) {
                    draw_line(ctx, p1[0], p1[1], p2[0], p2[1], '#FAFAFA', (1. - d/r)*3.);
                    n_pairs += 1
                }
            })
            draw_circle(ctx, p1[0], p1[1], .4*n_pairs + rmin, '#FFFFFF', '#FFFFFF');
        });
    }, controls];
}